import os
import numpy as np
from onnxruntime import InferenceSession, SessionOptions, GraphOptimizationLevel
os.environ["PHONEMIZER_ESPEAK_LIBRARY"] = os.path.abspath(os.path.join(os.getcwd(), r".\eSpeak NG\libespeak-ng.dll"))
os.environ["PHONEMIZER_ESPEAK_PATH"] = os.path.abspath(os.path.join(os.getcwd(), r".\eSpeak NG\espeak-ng.exe"))
from backbone import tokenize, phonemize
import time
import scipy.io.wavfile as wavfile

# Optimize ONNX Runtime session
session_options = SessionOptions()
session_options.graph_optimization_level = GraphOptimizationLevel.ORT_ENABLE_ALL
session_options.intra_op_num_threads = os.cpu_count()
# session_options.execution_mode = 1
session_options.enable_mem_pattern = True

model_name = 'model.onnx'
sess = InferenceSession(os.path.join('_tts', model_name), session_options, providers=["CUDAExecutionProvider"])



def generate_audio(text_input, file_name, speaking_speed=1, auto_pause=False, advanced=False, voice="af_nova", adjust_pitch=False, pitch_semitones=0):
    """
    Generate audio from text using Misaki TTS engine with improved auto-pause and optional pitch adjustment.
    
    Args:
        text_input (str): Text to convert to speech.
        file_name (str): Output file name (without extension).
        speaking_speed (float): Speech rate multiplier.
        auto_pause (bool): Whether to add natural, context-sensitive pauses.
        advanced (bool): Whether to use advanced pronunciation rules.
        voice (str): Voice model to use.
        adjust_pitch (bool): Whether to apply pitch adjustment to the generated audio.
        pitch_semitones (int): Number of semitones to shift pitch (positive or negative).
    """
    import os
    import numpy as np
    from scipy.io import wavfile

    # Apply advanced pronunciation adjustments if enabled.
    if advanced:
        text_input = apply_misaki_pronunciation(text_input)
        print("Advanced text:", text_input)
    
    # Handle newlines: replace them with sentence breaks.
    text_input = text_input.replace('\n', '. ')

    if auto_pause:
        # Split text into sentences and then segments.
        sentences = split_into_sentences(text_input)
        audio_segments = []
        
        for sentence in sentences:
            segments = split_sentence_into_segments(sentence)
            for seg_text, punctuation in segments:
                temp_file = "./.bin/audio_dumps/temp_segment"
                _generate_audio_segment(seg_text, voice, temp_file, speaking_speed, adjust_pitch, pitch_semitones)
                
                # Read generated audio
                _, seg_audio = wavfile.read(f"{temp_file}.wav")
                audio_segments.append(seg_audio)
                
                # Determine appropriate pause after the segment.
                pause_samples = determine_pause_length(seg_text, punctuation)
                if pause_samples > 0:
                    silence = np.zeros(pause_samples, dtype=np.int16)
                    audio_segments.append(silence)
                
                # Cleanup temporary segment file.
                if os.path.exists(f"{temp_file}.wav"):
                    os.remove(f"{temp_file}.wav")
        
        # Concatenate all segments and pauses.
        combined_audio = np.concatenate(audio_segments)
        wavfile.write(f"./.bin/audio_dumps/{file_name}.wav", 24000, combined_audio)
    else:
        temp_file = f"./.bin/audio_dumps/{file_name}"
        _generate_audio_segment(text_input, voice, temp_file, speaking_speed, adjust_pitch, pitch_semitones)

def _generate_audio_segment(text, voice, file_name, speaking_speed, adjust_pitch, pitch_semitones):
    """Internal function to generate a single audio segment and optionally adjust pitch."""
    import numpy as np
    from scipy.io import wavfile
    # Load voice model data.
    voices = np.fromfile(f'./_tts/voices/{voice}.bin', dtype=np.float32).reshape(-1, 1, 256)
    
    # Process text: phonemize and tokenize.
    phonemized_content = phonemize(text, lang='a')
    tokens = tokenize(phonemized_content)
    
    assert len(tokens) <= 510, f"Input too long: {len(tokens)} tokens"
    ref_s = voices[len(tokens)]
    tokens = [[0, *tokens, 0]]
    
    tokens = np.array(tokens, dtype=np.int64)
    ref_s = np.ascontiguousarray(ref_s, dtype=np.float32)
    speed = np.ones(speaking_speed, dtype=np.float32)
    
    # Generate the audio using your inference session.
    audio = sess.run(None, {
        'input_ids': tokens,
        'style': ref_s,
        'speed': speed,
    })[0]
    
    # If pitch adjustment is enabled, process audio to shift pitch.
    if adjust_pitch:
        try:
            import librosa
            # librosa expects float audio; shift pitch and then convert back.
            audio_float = audio[0].astype(np.float32)
            audio_shifted = librosa.effects.pitch_shift(audio_float, sr=24000, n_steps=pitch_semitones)
            audio[0] = audio_shifted.astype(np.int16)
        except ImportError:
            print("Librosa not installed; skipping pitch adjustment.")
    
    wavfile.write(f'{file_name}.wav', 24000, audio[0])

def split_into_sentences(text):
    """
    Split text into sentences preserving punctuation, handling abbreviations and newlines.
    Also considers ellipses as a distinct pause marker.
    """
    import re
    # Handle abbreviations separately.
    text = re.sub(r'(?<=[Mm]r)\.(?=\s)', '<PERIOD>', text)
    text = re.sub(r'(?<=[Dd]r)\.(?=\s)', '<PERIOD>', text)
    text = re.sub(r'(?<=[Mm]rs)\.(?=\s)', '<PERIOD>', text)
    text = re.sub(r'(?<=[Pp]rof)\.(?=\s)', '<PERIOD>', text)
    
    # Convert ellipses to a marker.
    text = text.replace("...", "<ELLIPSIS>")
    
    # Split at sentence boundaries (period, exclamation, question).
    sentences = re.split(r'(?<=[.!?])\s+', text)
    
    # Restore the actual periods and ellipsis markers.
    sentences = [s.replace('<PERIOD>', '.').replace('<ELLIPSIS>', '...') for s in sentences]
    
    return [s for s in sentences if s.strip()]

def split_sentence_into_segments(sentence):
    """
    Split a sentence into segments at punctuation boundaries.
    Takes into account colons used in time expressions.
    
    Returns a list of tuples: (segment_text, trailing_punctuation)
    """
    import re
    # Replace time-pattern colons with a marker to avoid mis-splitting.
    sentence = re.sub(r'(\d{1,2}):(\d{2})', r'\1<TIME_COLON>\2', sentence)
    
    # Split by punctuation, including comma, period, question, exclamation, semicolon, colon (if not time).
    pattern = r'([^,.!?;:]+)([,.!?;:]*)'
    segments = re.findall(pattern, sentence)
    
    # Restore time colons in each segment.
    cleaned_segments = []
    for seg, punc in segments:
        seg = seg.replace("<TIME_COLON>", ":").strip()
        punc = punc.strip()
        cleaned_segments.append((seg, punc))
    
    # Filter out empty segments.
    return [s for s in cleaned_segments if s[0]]

def determine_pause_length(segment_text, punctuation):
    """
    Determine dynamic pause length in samples (at 24kHz) based on punctuation and segment complexity.
    Additional pauses are added for ellipses.
    """
    # Base pause durations (seconds) for punctuation marks.
    base_durations = {
        '.': 0.5,
        '?': 0.6,
        '!': 0.6,
        ',': 0.2,
        ';': 0.3,
        ':': 0.3,
        '...': 0.7,
        '': 0.25  # Default pause.
    }
    # Set default duration.
    duration = 0.25
    if punctuation:
        # If multiple punctuation symbols, use the one with the highest base duration.
        for mark in punctuation:
            if mark in base_durations and base_durations[mark] > duration:
                duration = base_durations[mark]
    
    # Add extra pause for longer segments.
    words = segment_text.split()
    if len(words) > 10:
        extra = (len(words) - 10) * 0.02  # 20ms extra per extra word.
        duration += extra
    
    return int(duration * 24000)

def apply_misaki_pronunciation(text):
    """
    Apply Misaki phoneme pronunciation rules to text.
    Processes words with explicit phonetic transcription and attempts to generate pronunciation for others.
    """
    import re
    import phonemizer
    
    def replace_with_pronunciation(match):
        word = match.group(1)
        if match.group(2):
            return f"[{word}]({match.group(2)})"
        else:
            british = False  # Default to American pronunciation.
            espeak = phonemizer.backend.EspeakBackend(
                language=f"en-{'gb' if british else 'us'}",
                preserve_punctuation=True, with_stress=True, tie='^'
            )
            espeak_ps = espeak.phonemize([word])
            espeak_ps = espeak_ps[0].strip() if espeak_ps else ''
            ps = from_espeak(espeak_ps, british)
            return f"[{word}](/{ps}/)"
    
    formatted_pattern = r'\[([^\]]+)\](?:\(/([^/]+)/\))?'
    text = re.sub(formatted_pattern, replace_with_pronunciation, text)
    
    word_pattern = r'\b([A-Z][a-z]{2,})\b'
    def process_proper_noun(match):
        word = match.group(1)
        if f"[{word}]" in text:
            return word
        british = False
        espeak = phonemizer.backend.EspeakBackend(
            language=f"en-{'gb' if british else 'us'}",
            preserve_punctuation=True, with_stress=True, tie='^'
        )
        try:
            espeak_ps = espeak.phonemize([word])
            espeak_ps = espeak_ps[0].strip() if espeak_ps else ''
            ps = from_espeak(espeak_ps, british)
            return f"[{word}](/{ps}/)"
        except:
            return word
    text = re.sub(word_pattern, process_proper_noun, text)
    return text

def from_espeak(ps, british=False):
    """Convert espeak phonemes to Misaki phonemes."""
    import re
    FROM_ESPEAKS = sorted({
        '\u0303':'', 'a^ɪ':'I', 'a^ʊ':'W', 'd^ʒ':'ʤ', 'e':'A', 'e^ɪ':'A', 
        'r':'ɹ', 't^ʃ':'ʧ', 'x':'k', 'ç':'k', 'ɐ':'ə', 'ɔ^ɪ':'Y', 'ə^l':'ᵊl', 
        'ɚ':'əɹ', 'ɬ':'l', 'ʔ':'t', 'ʔn':'tᵊn', 'ʔˌn\u0329':'tᵊn', 'ʲ':'', 
        'ʲO':'jO', 'ʲQ':'jQ'
    }.items(), key=lambda kv: -len(kv[0]))
    
    for old, new in FROM_ESPEAKS:
        ps = ps.replace(old, new)
    ps = re.sub(r'(\S)\u0329', r'ᵊ\1', ps).replace(chr(809), '')
    
    if british:
        ps = ps.replace('e^ə', 'ɛː')
        ps = ps.replace('iə', 'ɪə')
        ps = ps.replace('ə^ʊ', 'Q')
    else:
        ps = ps.replace('o^ʊ', 'O')
        ps = ps.replace('ɜːɹ', 'ɜɹ')
        ps = ps.replace('ɜː', 'ɜɹ')
        ps = ps.replace('ɪə', 'iə')
        ps = ps.replace('ː', '')
    
    return ps.replace('^', '')



# inp = "Hello, this is a test of text to speech generation"
# total_time = 0
# iterations = 100

# for i in range(iterations):
#     start_time = time.time()
#     generate_audio(inp, "hf_beta")
#     total_time += time.time() - start_time

# average_time = total_time / iterations
# print("Average generation time: %s seconds" % average_time)


# inp = "Hello, my name is Echo! I am an advanced AI model that helps you convert text to speech."
# start = time.time()
# generate_audio(inp, "af_nova", "output.wav", auto_pause=True, advanced=True)
# end = time.time()
# print("Total time: %s seconds" % (end - start))





# import os

# def get_voice_files():
#     voice_dir = os.path.join("_tts", "voices")
#     voice_files = []
    
#     if os.path.exists(voice_dir):
#         for file in os.listdir(voice_dir):
#             if os.path.isfile(os.path.join(voice_dir, file)):
#                 voice_files.append(file)
    
#     return voice_files


# all_voices = get_voice_files()
# nation_list = {
#     "a": "american",
#     "b": "british",
#     "e": "spanish",
#     "f": "french",
#     "h": "hindi",
#     "i": "italian",
#     "j": "japanese",
#     "p": "portuguese",
#     "z": "chinese",
# }

# sample_meta = []
# import json

# for voice in all_voices:
#     name_of_voice = voice.split('.')[0]
#     print("\n\n")
#     try:
#         speaker_name = name_of_voice.split("_")[1]
#         print(speaker_name)
#         speaker_nationality = nation_list.get(name_of_voice.split("_")[0][0], "unknown")
#         print(speaker_nationality)
#         speaker_gender = "female" if name_of_voice.split("_")[0][1] == "f" else "male"
#         print(speaker_gender)
#         text = f"Hey there, I am {speaker_name}. I am a {speaker_nationality} {speaker_gender} voice. I can convert text to speech."
#     except Exception as e:
#         speaker_name = "default"
#         print(speaker_name)
#         speaker_nationality = "unknown"
#         print(speaker_nationality)
#         speaker_gender = "female"
#         print(speaker_gender)
#         text = "Hey there, I am the default voice of Echo. I can convert text to speech."
#     gen_audio_file_name = f"voices/{name_of_voice}"
#     generate_audio(text, gen_audio_file_name, voice=name_of_voice, auto_pause=True)
#     data = {
#         "name": speaker_name,
#         "gender": speaker_gender,
#         "nationality": speaker_nationality,
#         "voice_file": voice,
#         "voice_id": name_of_voice,
#         "voice": "https://voices.echo.brahmai.in/samples/" + voice,
#     }

#     sample_meta.append(data)

# with open("./samples.json", "w") as f:
#     json.dump(sample_meta, f, indent=4)