# プロンプトチューニング

<CourseFloatingBanner chapter={10}
  classNames="absolute z-10 right-0 top-0"
  notebooks={[
    {label: "Google Colab", value: "https://colab.research.google.com/github/huggingface/smol-course/blob/main/notebooks/ja/3_parameter_efficient_finetuning/finetune_sft_peft.ipynb"},
]} />
    
プロンプトチューニングは、モデルの重みではなく入力表現を変更するパラメータ効率の良いアプローチです。従来の微調整がすべてのモデルパラメータを更新するのに対し、プロンプトチューニングはベースモデルを固定したまま、少数の学習可能なトークンを追加して最適化します。

## プロンプトチューニングの理解

プロンプトチューニングは、学習可能な連続ベクトル（ソフトプロンプト）を入力テキストの前に追加するパラメータ効率の良い微調整方法です。従来の離散テキストプロンプトとは異なり、これらのソフトプロンプトはベースモデルを固定したまま、逆伝播を通じて学習されます。この方法は、["The Power of Scale for Parameter-Efficient Prompt Tuning"](https://huggingface.co/papers/2104.08691)（Lester et al., 2021）で紹介され、モデルサイズが大きくなるにつれてプロンプトチューニングがモデル微調整と競争力を持つようになることが示されました。論文内では、約100億パラメータのモデルで、プロンプトチューニングがモデル微調整の性能に匹敵し、タスクごとに数百のパラメータしか変更しないことが示されています。

これらのソフトプロンプトは、トレーニング中に最適化されるモデルの埋め込み空間内の連続ベクトルです。従来の自然言語トークンを使用する離散プロンプトとは異なり、ソフトプロンプトは固有の意味を持たず、勾配降下を通じて固定モデルから望ましい動作を引き出すことを学習します。この技術は、各タスクに対して小さなプロンプトベクトル（通常は数百パラメータ）を保存するだけで済むため、マルチタスクシナリオに特に効果的です。このアプローチは、最小限のメモリフットプリントを維持するだけでなく、モデルの再読み込みなしにプロンプトベクトルを交換するだけで迅速なタスク切り替えを可能にします。

## トレーニングプロセス

ソフトプロンプトは通常、8〜32トークンで構成され、ランダムに初期化するか、既存のテキストから初期化されます。初期化方法はトレーニングプロセスにおいて重要な役割を果たし、テキストベースの初期化はランダム初期化よりも優れた性能を発揮することがよくあります。

トレーニング中は、プロンプトパラメータのみが更新され、ベースモデルは固定されたままです。この集中アプローチは標準的なトレーニング目標を使用しますが、プロンプトトークンの学習率と勾配の挙動に注意を払う必要があります。

## PEFTを使用した実装

PEFTライブラリを使用すると、プロンプトチューニングの実装が簡単になります。以下は基本的な例です：

```python
from peft import PromptTuningConfig, TaskType, get_peft_model
from transformers import AutoModelForCausalLM, AutoTokenizer

# ベースモデルを読み込む
model = AutoModelForCausalLM.from_pretrained("your-base-model")
tokenizer = AutoTokenizer.from_pretrained("your-base-model")

# プロンプトチューニングを設定
peft_config = PromptTuningConfig(
    task_type=TaskType.CAUSAL_LM,
    num_virtual_tokens=8,  # 学習可能なトークンの数
    prompt_tuning_init="TEXT",  # テキストから初期化
    prompt_tuning_init_text="このテキストがポジティブかネガティブかを分類してください：",
    tokenizer_name_or_path="your-base-model",
)

# プロンプトチューニング可能なモデルを作成
model = get_peft_model(model, peft_config)
```

## 他の方法との比較

他のPEFTアプローチと比較すると、プロンプトチューニングはその効率性で際立っています。LoRAは低パラメータ数とメモリ使用量を提供しますが、タスク切り替えにはアダプタの読み込みが必要です。プロンプトチューニングはさらに低いリソース使用量を達成し、即時のタスク切り替えを可能にします。対照的に、フルファインチューニングは多くのリソースを必要とし、異なるタスクごとに別々のモデルコピーが必要です。

| 方法 | パラメータ | メモリ | タスク切り替え |
|--------|------------|---------|----------------|
| プロンプトチューニング | 非常に低い | 最小限 | 簡単 |
| LoRA | 低い | 低い | 読み込みが必要 |
| フルファインチューニング | 高い | 高い | 新しいモデルコピー |

プロンプトチューニングを実装する際は、最初に少数の仮想トークン（8〜16）を使用し、タスクの複雑さが要求する場合にのみ増やします。タスクに関連するテキストを使用したテキスト初期化は、ランダム初期化よりも優れた結果をもたらすことがよくあります。初期化戦略は、ターゲットタスクの複雑さを反映する必要があります。

トレーニングには、フルファインチューニングとは異なる考慮事項が必要です。高い学習率が効果的なことが多いですが、プロンプトトークンの勾配を注意深く監視することが重要です。多様な例で定期的に検証することで、さまざまなシナリオでの堅牢な性能を確保します。

## 応用

プロンプトチューニングは、次のようなシナリオで優れた効果を発揮します：

1. マルチタスク展開
2. リソース制約のある環境
3. 迅速なタスク適応
4. プライバシーに敏感なアプリケーション

モデルが小さくなるにつれて、プロンプトチューニングはフルファインチューニングと比較して競争力が低下します。例えば、SmolLM2のようなモデルでは、プロンプトチューニングはフルファインチューニングよりも関連性が低くなります。

## 次のステップ

⏭️ [LoRAアダプタのチュートリアル](../../../notebooks/ja/3_parameter_efficient_finetuning/finetune_sft_peft.ipynb)に進み、LoRAアダプタでモデルを微調整する方法を学びましょう。

## リソース
- [PEFTドキュメント](https://huggingface.co/docs/peft)
- [プロンプトチューニング論文](https://huggingface.co/papers/2104.08691)
- [Hugging Face Cookbook](https://huggingface.co/learn/cookbook/prompt_tuning_peft)


<EditOnGithub source="https://github.com/huggingface/smol-course/blob/main/units/ja/unit3/3.md" />