# core.py
import threading
import time
from pystray import Icon, MenuItem, Menu
from PIL import Image, ImageDraw, ImageFont
from functools import partial
import os
import wmi
import sys
import colorsys
import pythoncom
import ctypes
import ctypes.wintypes
import win32gui
import win32con
import pynvml
from pynvml import nvmlDeviceGetTemperature, nvmlDeviceGetTemperatureThreshold, NVML_TEMPERATURE_GPU, NVML_TEMPERATURE_THRESHOLD_GPU_MAX, nvmlInit, nvmlShutdown, nvmlDeviceGetHandleByIndex, nvmlDeviceGetUtilizationRates, nvmlDeviceGetMemoryInfo, nvmlDeviceGetCount
import psutil
from difflib import get_close_matches

                   


# Farben
COLOR_MAP = {
    "gray": (160, 160, 160, 255),
    "green": (0, 128, 0, 255),
    "red": (128, 0, 0, 255),
    "yellow": (192, 192, 0, 255)
}


# Global
icons = {}
current_colors = {}
last_colors = {}
stop_events = {}
color_lock = threading.Lock()


# Special font
def resource_path(relative_path):
    try:
        # Wenn über PyInstaller ausgeführt
        base_path = sys._MEIPASS
    except AttributeError:
        # Beim normalen Ausführen (z. B. im Editor)
        base_path = os.path.abspath(".")
    return os.path.join(base_path, relative_path)

font_path = resource_path("fonts/DePixelSchmal.otf")


# anzeige 5% schritte reduziert
def round_to_nearest_five(value):
    return int(round(value / 5.0) * 5)


# netzwerk monitor definieren
def find_best_match(name, candidates):
    matches = get_close_matches(name, candidates, n=1, cutoff=0.6)
    return matches[0] if matches else None



def get_active_network_adapters():
    c = wmi.WMI()
    adapters = []
    for nic in c.Win32_NetworkAdapterConfiguration(IPEnabled=True):
        if hasattr(nic, 'Description'):
            adapters.append(nic.Description)
    return adapters

def get_adapter_speeds():
    c = wmi.WMI()
    speeds = {}
    for nic in c.Win32_NetworkAdapter():
        if nic.NetEnabled and nic.Speed:
            speeds[nic.Name] = int(nic.Speed)  # Bits per second
    return speeds


# icon Netzwerk
def create_text_icon(text, color=(255, 255, 255, 255), bg_color=(0, 0, 0, 0)):
    size = 77
    image = Image.new("RGBA", (size, size), bg_color)
    draw = ImageDraw.Draw(image)

    try:
        font = ImageFont.truetype(font_path, 30)
    except Exception:
        font = ImageFont.load_default()

    try:
        bbox = draw.textbbox((0, 0), text, font=font)
        text_w, text_h = bbox[2] - bbox[0], bbox[3] - bbox[1]
        text_x = (size - text_w) // 2 - bbox[0]
        text_y = (size - text_h) // 2 - bbox[1]
    except AttributeError:
        text_w, text_h = font.getsize(text)
        text_x = (size - text_w) // 2
        text_y = (size - text_h) // 2

    draw.text((text_x, text_y), text, font=font, fill=color)
    return image


def format_speed_custom(value_kb):
    """
    Gibt einen kompakten Text im Format "0.9kB/s", "10MB/s", "99GB/s" aus.
    Automatischer Einheitenwechsel ab 99.
    """
    units = ['kB/s', 'MB/s', 'GB/s']
    speed = value_kb
    unit_index = 0

    # Automatischer Einheitenwechsel ab 100 kB/s
    while speed >= 100 and unit_index < len(units) - 1:
        speed /= 1024
        unit_index += 1

    # Darstellung mit maximal 2 Ziffern
    if speed < 10:
        display = f"{speed:.1f}"  # z.B. 0.9, 1.8
    else:
        display = f"{min(round(speed), 99)}"  # z.B. 10, 55, 99 (aber nie 100)

    return f"{display}\n{units[unit_index]}"



# -------------------------------------------------------------
# WMI: Zuordnung physischer Laufwerke zu logischen Partitionen
# -------------------------------------------------------------
def get_physical_drives_with_partitions():
    c = wmi.WMI()
    drive_map = {}

    for disk in c.Win32_DiskDrive():
        disk_id = disk.DeviceID.split("\\")[-1].upper()
        if disk_id not in drive_map:
            drive_map[disk_id] = []

        partitions = disk.associators("Win32_DiskDriveToDiskPartition")
        for partition in partitions:
            logical_disks = partition.associators("Win32_LogicalDiskToPartition")
            for logical_disk in logical_disks:
                letter = logical_disk.DeviceID.upper().strip()
                if letter not in drive_map[disk_id]:
                    drive_map[disk_id].append(letter)

    return drive_map

# -------------------------------------------------------------
# Tray-Icon Erstellung und Verwaltung
# -------------------------------------------------------------

# Hard Drive Icon Color
def get_color(read_active, write_active, read_mb=0, write_mb=0):
    # Beide unter 2MB/s Schwellwert -> grau
    if read_mb < 2 and write_mb < 2:
        return "gray"
    
    # Beide aktiv über Schwellwert
    if read_mb >= 2 and write_mb >= 2:
        ratio = read_mb / write_mb if write_mb != 0 else float('inf')
        if 1/3 <= ratio <= 3:
            return "yellow"
        elif write_mb > read_mb:
            return "red"
        else:
            return "green"
    
    # Nur schreiben aktiv
    if write_mb >= 2:
        return "red"
    
    # Nur lesen aktiv
    if read_mb >= 2:
        return "green"
    
    return "gray"

def _set_icon_color(key, color):
    icon_data = icons.get(key)
    if icon_data:
        icon = icon_data["icon"]
        label = icon_data["label"]
        new_icon = create_icon(COLOR_MAP.get(color, (128, 128, 128)), label)
        icon.icon = new_icon
        with color_lock:
            last_colors[key] = color

# center letter
def create_icon(color_rgb, label):
    size = 77
    image = Image.new("RGBA", (size, size), (0, 0, 0, 0))
    draw = ImageDraw.Draw(image)

    #draw.ellipse((0, 0, size, size), fill=color_rgb + (255,))
    draw.ellipse((0, 0, size, size), fill=color_rgb)
    try:
        font = ImageFont.truetype(font_path, 60)
    except Exception:
        font = ImageFont.load_default()

    try:
        bbox = draw.textbbox((0, 0), label, font=font)
        text_w, text_h = bbox[2] - bbox[0], bbox[3] - bbox[1]
    except AttributeError:
        text_w, text_h = font.getsize(label)

    text_x = (size - text_w) / 2
    text_y = (size - text_h) / 2 - bbox[1]


    brightness = sum(color_rgb) / 3
    text_color = (0, 0, 0, 255) if brightness > 130 else (255, 255, 255, 255)

    draw.text((text_x, text_y), label, font=font, fill=text_color)
    return image



def update_tray_color(key, color):
    with color_lock:
        old_color = current_colors.get(key)
        if old_color == color:
            return
        current_colors[key] = color

    _set_icon_color(key, color)

def _icon_updater(key, stop_event):
    print(f"[DEBUG] _icon_updater gestartet für {key}")
    while not stop_event.is_set():
        with color_lock:
            color = current_colors.get(key, "gray")
            last_color = last_colors.get(key)

        if color != last_color:
            _set_icon_color(key, color)
        stop_event.wait(0.2)

def _on_quit(icon_inst, item, callback):
    icon_inst.visible = False
    icon_inst.stop()
    callback()
    os._exit(0)


# Sortierung Z-A im Tray dann A-Z
def sort_device_partitions(device_map):
    # Alle (device, partition)-Tupel extrahieren
    items = [(dev, part) for dev, parts in device_map.items() for part in parts]
    # Sortierung ausschließlich nach Partition-Buchstabe, absteigend (Z → A)
    sorted_items = sorted(items, key=lambda x: x[1].upper(), reverse=True)
    return sorted_items



# Zuordnung Tray
def start_tray_icons(on_quit_callback, device_map):
    on_quit_callback = lambda: stop_all_tray_icons()
    final_sorted_items = sort_device_partitions(device_map)

    for index, (dev, part) in enumerate(final_sorted_items):
        icon_label = part.strip(":")
        icon_key = f"{dev}_{part}"
        icon_title = f"{index:02d}_SmartDisk_{icon_label}"

        icon_image = create_icon(COLOR_MAP["gray"], icon_label)
        icon = Icon(icon_title, icon_image, menu=Menu(MenuItem("Exit", lambda icon_inst, item: _on_quit(icon_inst, item, on_quit_callback))))
        icons[icon_key] = {
            "icon": icon,
            "label": icon_label
        }
        current_colors[icon_key] = "gray"
        last_colors[icon_key] = None

        stop_event = threading.Event()
        stop_events[icon_key] = stop_event

        icon.run_detached()

        threading.Thread(target=_icon_updater, args=(icon_key, stop_event), daemon=True).start()
        time.sleep(0.2)


# farb-balken
def get_gradient_color(percent):
    """
    Liefert eine Farbe zwischen grün, gelb und rot für 0-100%.
    - 0% = dunkelgrün (0, 128, 0)
    - 50% = dunkelgelb (128, 128, 0)
    - 100% = dunkelrot (128, 0, 0)
    """
    if percent < 50:
        # grün → gelb
        ratio = percent / 50
        r = int(0 + (128 - 0) * ratio)
        g = 128
        b = 0
    else:
        # gelb → rot
        ratio = (percent - 50) / 50
        r = 128
        g = int(128 - (128 * ratio))
        b = 0
    return (r, g, b, 255)


# ICON  CPU,RAM,VRAM,GPU,Temp
def create_bar_icon(percent, label, color=None):
    if color is None:
        color = get_gradient_color(percent)
    #64/28
    size = 77
    image = Image.new("RGBA", (size, size), (0, 0, 0, 0))
    draw = ImageDraw.Draw(image)

    # Rahmen zeichnen
    draw.rectangle([0, 0, size - 1, size - 1], outline=(100, 100, 100, 255))

    # Füllhöhe berechnen (von unten)
    bar_height = int(size * (percent / 100))
    top = size - bar_height
    draw.rectangle([0, top, size, size], fill=color)

    try:
        font = ImageFont.truetype(font_path, 30)
    except:
        font = ImageFont.load_default()

    draw.text((2, 2), label, font=font, fill=(255, 255, 255, 255))
    return image



def stop_all_tray_icons():
    for icon in icons.values():
        try:
            if icon["icon"].visible:
                icon["icon"].visible = False
                icon["icon"].stop()
        except Exception as e:
            print(f"[WARN] Icon-Stop fehlgeschlagen: {e}")
    for event in stop_events.values():
        event.set()


# tool tip
def update_tray_tooltip(key, tooltip_text):
    """
    Setzt den Tooltip (Mouseover-Hinweis) für das Tray-Icon.
    """
    icon_data = icons.get(key)
    if icon_data:
        icon = icon_data["icon"]
        try:
            icon.title = tooltip_text
        except Exception as e:
            print(f"[WARN] Tooltip konnte nicht gesetzt werden für {key}: {e}")



# Disk-Monitoring über WMI

READ_WRITE_THRESHOLD = 1024 * 1024 * 2  # 2 MB


# GPU , VRAM, Temperartur
class MultiGpuMonitor(threading.Thread):
    def __init__(self, poll_interval=1):
        super().__init__()
        self.poll_interval = poll_interval
        self.running = True

    def run(self):
        nvmlInit()
        try:
            device_count = nvmlDeviceGetCount()
            if device_count == 0:
                print("[INFO] Keine NVIDIA-GPUs erkannt.")
                return

            self.handles = [
                nvmlDeviceGetHandleByIndex(i)
                for i in range(device_count)
            ]

            while self.running:
                for idx, handle in enumerate(self.handles):
                    util = nvmlDeviceGetUtilizationRates(handle)
                    mem = nvmlDeviceGetMemoryInfo(handle)

                    gpu_util = round_to_nearest_five(util.gpu)
                    vram_util = round_to_nearest_five(mem.used / mem.total * 100)

                    # GPU-Icon
                    key_gpu = f"GPU{idx}_USAGE"
                    label_gpu = f"GPU{idx}"
                    image_gpu = create_bar_icon(gpu_util, label_gpu)
                    if key_gpu not in icons:
                        icon = Icon(key_gpu, image_gpu, menu=Menu(MenuItem("Exit", lambda icon_inst, item: _on_quit(icon_inst, item, on_quit_callback))))
                        #icon = Icon(key_gpu, image_gpu)
                        icons[key_gpu] = {"icon": icon, "label": label_gpu}
                        icon.run_detached()
                    else:
                        icons[key_gpu]["icon"].icon = image_gpu
                    icons[key_gpu]["icon"].title = f"{label_gpu} {gpu_util}%"

                    # VRAM-Icon
                    key_vram = f"VRAM{idx}_USAGE"
                    label_vram = f"VR{idx}"
                    image_vram = create_bar_icon(vram_util, label_vram)
                    if key_vram not in icons:
                        icon = Icon(key_vram, image_vram, menu=Menu(MenuItem("Exit", lambda icon_inst, item: _on_quit(icon_inst, item, on_quit_callback))))
                        #icon = Icon(key_vram, image_vram)
                        icons[key_vram] = {"icon": icon, "label": label_vram}
                        icon.run_detached()
                    else:
                        icons[key_vram]["icon"].icon = image_vram

                    used_gb = round(mem.used / (1024**3))
                    total_gb = round(mem.total / (1024**3))
                    icons[key_vram]["icon"].title = f"VRAM {used_gb}/{total_gb} GB"


                    # --- Temperatur auslesen & normieren ---
                    temp = nvmlDeviceGetTemperature(handle, NVML_TEMPERATURE_GPU)
                    try:
                        max_temp = nvmlDeviceGetTemperatureThreshold(
                            handle,
                            NVML_TEMPERATURE_THRESHOLD_GPU_MAX
                        )
                    except Exception:
                        max_temp = 95  # Fallback

                    clamped = max(35, min(temp, max_temp))
                    pct = int((clamped - 35) / (max_temp - 35) * 100)

                    # --- Tray-Icon erzeugen ---
                    key_temp = f"GPU{idx}_TEMP"
                    label_temp = f"T{idx}"
                    image_temp = create_bar_icon(pct, label_temp, color=(128, 0, 0, 255))

                    if key_temp not in icons:
                        icon = Icon(key_temp, image_temp, menu=Menu(MenuItem("Exit", lambda icon_inst, item: _on_quit(icon_inst, item, on_quit_callback))))
                        #icon = Icon(key_temp, image_temp)
                        icons[key_temp] = {"icon": icon, "label": label_temp}
                        icon.run_detached()
                    else:
                        icons[key_temp]["icon"].icon = image_temp

                    icons[key_temp]["icon"].title = f"{label_temp}: {temp} °C"
                    


                time.sleep(self.poll_interval)

        except Exception as e:
            print(f"[ERROR] GPU Monitoring Fehler: {e}")
        finally:
            nvmlShutdown()

    def stop(self):
        self.running = False



# CPU-Monitoring über psutil
class CpuMonitor(threading.Thread):
    def __init__(self, update_cpu_icon, poll_interval=1):
        super().__init__()
        self.update_cpu_icon = update_cpu_icon
        self.poll_interval = poll_interval
        self.running = True

    def run(self):
        while self.running:
            try:
                cpu_util = psutil.cpu_percent(interval=None)
                self.update_cpu_icon(int(cpu_util))
            except Exception as e:
                print(f"[ERROR] CPU Monitoring Fehler: {e}")
            time.sleep(self.poll_interval)

    def stop(self):
        self.running = False


# RAM-Monitoring über psutil
class RamMonitor(threading.Thread):
    def __init__(self, update_ram_icon, poll_interval=1):
        super().__init__()
        self.update_ram_icon = update_ram_icon
        self.poll_interval = poll_interval
        self.running = True

    def run(self):
        while self.running:
            try:
                mem = psutil.virtual_memory()
                ram_util = int(mem.percent)
                self.update_ram_icon(ram_util)
            except Exception as e:
                print(f"[ERROR] RAM Monitoring Fehler: {e}")
            time.sleep(self.poll_interval)

    def stop(self):
        self.running = False


# WMI call network and disks
class UnifiedWmiMonitor(threading.Thread):
    def __init__(self, disk_callbacks, net_callback, device_partitions, poll_interval=1):
        super().__init__()
        self.disk_callbacks = disk_callbacks
        self.net_callback = net_callback
        self.device_partitions = device_partitions
        self.poll_interval = poll_interval
        self.running = True
        self.prev_disk_counters = {}
        self.prev_net_stats = {}
        self.speeds = get_adapter_speeds()
        
        # Automatisch erstellen
        self.active_adapters = get_active_network_adapters()
        self.adapter_map = {}
        for active in self.active_adapters:
            for candidate in self.speeds:
                if find_best_match(active, [candidate]) == candidate:
                    self.adapter_map[active] = candidate        

    def run(self):
        pythoncom.CoInitialize()
        c = wmi.WMI(namespace="root\\CIMV2")

        try:
            while self.running:
                try:
                    # --------- Disks ---------
                    perf_logical_disks = {
                        disk.Name.upper(): disk
                        for disk in c.Win32_PerfRawData_PerfDisk_LogicalDisk()
                    }

                    for dev, part in self.device_partitions:
                        perf_disk = perf_logical_disks.get(part)
                        if not perf_disk:
                            continue

                        read_bytes = int(perf_disk.DiskReadBytesPerSec)
                        write_bytes = int(perf_disk.DiskWriteBytesPerSec)
                        prev_read, prev_write = self.prev_disk_counters.get(part, (0, 0))
                        read_diff = max(0, read_bytes - prev_read)
                        write_diff = max(0, write_bytes - prev_write)
                        self.prev_disk_counters[part] = (read_bytes, write_bytes)

                        mb_read = read_diff / 1024 / 1024 / self.poll_interval
                        mb_write = write_diff / 1024 / 1024 / self.poll_interval

                        key = f"{dev}_{part}"
                        color = get_color(mb_read > 2.0, mb_write > 2.0, mb_read, mb_write)

                        if key in self.disk_callbacks:
                            self.disk_callbacks[key](color)
                            update_tray_tooltip(key, f"{part} R {int(mb_read)} MB/s | W {int(mb_write)} MB/s")

                    # --------- Netzwerk ---------
                    perf_net_ifaces = c.Win32_PerfRawData_Tcpip_NetworkInterface()

                    for iface in perf_net_ifaces:
                        iface_name = getattr(iface, 'Name', None)
                        if not iface_name:
                            continue

                        adapter_name = find_best_match(iface_name, list(self.adapter_map.keys()))
                        if not adapter_name:
                            continue

                        send = int(iface.BytesSentPersec)
                        recv = int(iface.BytesReceivedPersec)

                        prev_send, prev_recv = self.prev_net_stats.get(adapter_name, (0, 0))
                        send_diff = max(0, send - prev_send)
                        recv_diff = max(0, recv - prev_recv)

                        self.prev_net_stats[adapter_name] = (send, recv)

                        send_kb = send_diff / 1024
                        recv_kb = recv_diff / 1024

                        if self.net_callback:
                            self.net_callback(adapter_name, send_kb, recv_kb)

                except Exception as e:
                    print(f"[ERROR] Unified WMI Monitor: {e}")

                time.sleep(self.poll_interval)
        finally:
            pythoncom.CoUninitialize()

    def stop(self):
        self.running = False





# Start
def start_core_system():
    # -----------------------------
    # ZUERST: CPU / VRAM / GPU / RAM - nach erstem symbol orden sich alle anderen symbole als zweiter ein
    # -----------------------------
    on_quit_callback = lambda: stop_all_tray_icons()
    #menu=create_exit_menu(on_quit_callback)
    
    def update_cpu(percent):
        percent = round_to_nearest_five(percent)
        image = create_bar_icon(percent, "CPU")
        key = "CPU_USAGE"
        if key not in icons:
            icon = Icon("CPU_Usage", image, menu=Menu(MenuItem("Exit", lambda icon_inst, item: _on_quit(icon_inst, item, on_quit_callback))))
            #icon = Icon("CPU_Usage", image, menu=None)
            icons[key] = {"icon": icon, "label": "CPU"}
            icon.run_detached()
        else:
            icons[key]["icon"].icon = image
        icons[key]["icon"].title = f"CPU {percent}%"


    def update_ram(percent):
        percent = round_to_nearest_five(percent)
        image = create_bar_icon(percent, "RAM")
        key = "RAM_USAGE"
        if key not in icons:
            icon = Icon("RAM_Usage", image, menu=Menu(MenuItem("Exit", lambda icon_inst, item: _on_quit(icon_inst, item, on_quit_callback))))
            #icon = Icon("RAM_Usage", image, menu=None)
            icons[key] = {"icon": icon, "label": "RAM"}
            icon.run_detached()
        else:
            icons[key]["icon"].icon = image
        try:
            mem = psutil.virtual_memory()
            used_gb = round(mem.used / (1024 ** 3))
            total_gb = round(mem.total / (1024 ** 3))
            tooltip = f"RAM {used_gb} / {total_gb} GB"
        except Exception:
            tooltip = f"RAM {percent}%"
        icons[key]["icon"].title = tooltip



    # CPU starten
    cpu_monitor = CpuMonitor(update_cpu)
    cpu_monitor.daemon = True
    cpu_monitor.start()
    time.sleep(0.2)



    def update_net_icons(adapter_name, send_kb, recv_kb):

        def update_icon(direction, value_kb):
            # Werte unter 10 kB/s als 0 anzeigen
            if value_kb < 10:
                value_kb = 0

            key = f"NET_{adapter_name}_{direction}"

            # Darstellungstext mit Zeilenumbruch (für Icon)
            text_with_linebreak = f"{'U:' if direction == 'SEND' else 'D:'}{format_speed_custom(value_kb)}"

            # Darstellungstext ohne Zeilenumbruch (für Tooltip)
            # Format_speed_custom liefert z.B. "0.9\nMB/s"
            speed_parts = format_speed_custom(value_kb).split("\n")
            text_no_linebreak = f"{speed_parts[0]} {speed_parts[1]}" if len(speed_parts) == 2 else format_speed_custom(value_kb)

            image = create_text_icon(text_with_linebreak)

            # Icon erzeugen oder aktualisieren
            if key not in icons:
                icon = Icon(key, image, menu=Menu(MenuItem("Exit", lambda icon_inst, item: _on_quit(icon_inst, item, on_quit_callback))))
                #icon = Icon(key, image, menu=None)
                icons[key] = {"icon": icon, "label": key}
                icon.run_detached()
            else:
                icons[key]["icon"].icon = image

            # Tooltip setzen (ohne Zeilenumbruch)
            icons[key]["icon"].title = f"{adapter_name} {'Upload' if direction == 'SEND' else 'Download'}: {text_no_linebreak}"

        # Beide Richtungen unabhängig behandeln
        threading.Thread(target=update_icon, args=("SEND", send_kb), daemon=True).start()
        threading.Thread(target=update_icon, args=("RECV", recv_kb), daemon=True).start()


    # Multi-GPU starten lnkl Temp
    gpu_monitor = MultiGpuMonitor()
    gpu_monitor.daemon = True
    gpu_monitor.start()
    time.sleep(0.2)


    # RAM starten
    ram_monitor = RamMonitor(update_ram)
    ram_monitor.daemon = True
    ram_monitor.start()
    time.sleep(0.2)

    # ---------- Laufwerke ----------
    drive_map = get_physical_drives_with_partitions()
    if not drive_map:
        raise RuntimeError("Keine physischen Laufwerke gefunden!")

    tray_callbacks = {
        f"{dev}_{part}": partial(update_tray_color, f"{dev}_{part}")
        for dev, parts in drive_map.items()
        for part in parts
    }

    flat = [(dev, part) for dev, parts in drive_map.items() for part in parts]

    # Unified Monitor starten (disk+net)
    wmi_monitor = UnifiedWmiMonitor(tray_callbacks, update_net_icons, flat)
    wmi_monitor.daemon = True
    wmi_monitor.start()
    time.sleep(0.2)

    # Tray-Icons für Platten verzögert (nach GPU, RAM usw.)
    def delayed_drive_icons():
        time.sleep(2)
        start_tray_icons(lambda: None, drive_map)

    threading.Thread(target=delayed_drive_icons, daemon=True).start()

    return drive_map



    

