"""
Example workflow demonstrating complete pipeline usage
"""
from .unified_controller import UnifiedPipelineController
from typing import Any, Dict

def run_complete_pipeline(input_data: Any,
                        input_type: str = None,
                        output_type: str = None,
                        visualization: str = None) -> Dict[str, Any]:
    """
    Run complete pipeline from input to visualization
    
    Flow:
    1. Initialize pipeline
    2. Process input
    3. Execute model
    4. Process output
    5. Generate visualization
    
    Args:
        input_data: Raw input (text, image, audio, video, tensor)
        input_type: Type of input (auto-detected if None)
        output_type: Desired output format
        visualization: Type of visualization to generate
        
    Returns:
        Dict containing processed output and optional visualization
    """
    # Initialize unified pipeline
    controller = UnifiedPipelineController()
    
    try:
        # Process input through tensor pipeline
        print("Processing input...")
        input_addr = controller.process_input(input_data, input_type)
        
        # Execute model operations
        print("Executing model...")
        output_addr = controller.execute_model(input_addr)
        
        # Process and visualize output
        print("Processing output...")
        result = controller.process_output(
            output_addr,
            output_type or 'tensor',
            visualization
        )
        
        # Get pipeline status
        status = controller.get_pipeline_status()
        
        return {
            'status': 'success',
            'result': result,
            'pipeline_status': status
        }
        
    except Exception as e:
        return {
            'status': 'error',
            'error': str(e),
            'pipeline_status': controller.get_pipeline_status()
        }

# Example usage
if __name__ == "__main__":
    # Text input example
    text_result = run_complete_pipeline(
        "Hello, how are you?",
        input_type='text',
        output_type='text',
        visualization='line'  # Plot token probabilities
    )
    print("Text Pipeline Result:", text_result)
    
    # Image input example
    image_result = run_complete_pipeline(
        "input_image.jpg",
        input_type='image',
        output_type='image',
        visualization='heatmap'  # Attention visualization
    )
    print("Image Pipeline Result:", image_result)
    
    # Audio input example
    audio_result = run_complete_pipeline(
        "input_audio.wav",
        input_type='audio',
        output_type='audio',
        visualization='spectrogram'  # Audio visualization
    )
    print("Audio Pipeline Result:", audio_result)
