"""
Hash computation utilities optimized for tensor operations
"""
import numpy as np
from typing import Dict, Any, Optional, List
from ..core.sha256_matrix_ops import SHA256MatrixOps

class HashComputer:
    def __init__(self):
        self.sha256_ops = SHA256MatrixOps()
        self.target = np.ones(32, dtype=np.uint32) * 0xFFFFFFFF  # Accept any hash for personal mining
        
    def compute_hash(self, work_unit: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """Compute hashes for a work unit using tensor operations"""
        block_matrix = work_unit['block_matrix']
        nonce_range = work_unit['nonce_range']
        
        # Search for valid nonces in parallel
        valid_nonce, hash_result = self.sha256_ops.parallel_nonce_search(
            block_matrix,
            self.target,
            nonce_range
        )
        
        if valid_nonce is not None:
            return {
                'nonce': valid_nonce,
                'hash': hash_result[0],  # Get hash for the valid nonce
                'timestamp': np.uint32(time.time())
            }
        return None
        
    def compute_hash_batch(self, work_unit: Dict[str, Any]) -> List[Dict[str, Any]]:
        """Compute hashes for multiple blocks in parallel"""
        results = []
        chunk_size = 1024 * 1024 * 5  # Process 5M blocks per chunk
        
        for i in range(0, work_unit['block_matrix'].shape[0], chunk_size):
            chunk = {
                'block_matrix': work_unit['block_matrix'][i:i+chunk_size],
                'nonce_range': work_unit['nonce_range']
            }
            result = self.compute_hash(chunk)
            if result:
                results.append(result)
                
        return results
