"""
Simplified mining validation without traditional Bitcoin constraints
"""
from typing import Dict, Any
import hashlib

class MiningValidator:
    def __init__(self):
        # Simplified fixed difficulty target
        self.target = int("00FF0000" + "F" * 56, 16)  # Easier target for faster mining
        
    def verify_hash(self, block_hash: str) -> bool:
        """Verify if hash meets target difficulty"""
        hash_int = int(block_hash, 16)
        return hash_int < self.target
        
    def verify_block_structure(self, block: Dict[str, Any]) -> bool:
        """Verify basic block structure"""
        required_fields = ['version', 'prev_block_hash', 'timestamp', 'nonce', 'miner_address']
        return all(field in block for field in required_fields)
        
    def verify_block_hash(self, block: Dict[str, Any]) -> bool:
        """Verify block hash is correct"""
        # Reconstruct block header
        header = (
            str(block['version']).encode() +
            block['prev_block_hash'].encode() +
            str(block['timestamp']).encode() +
            str(block['nonce']).encode() +
            block['miner_address'].encode()
        )
        computed_hash = hashlib.sha256(header).hexdigest()
        return computed_hash == block.get('hash', '')
