import sys
import os

sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '../../src')))

from driver_api import VirtualGPUDriver

def run_simple_graphics_example():
    driver = VirtualGPUDriver()
    
    try:
        # Initialize the driver with minimal hardware for demonstration
        driver.initialize(num_chips=1, vram_size_gb=0.001, num_sms_per_chip=1, num_cores_per_sm=1, num_tensor_cores_per_sm=1)

        # Create a framebuffer
        framebuffer_width = 16
        framebuffer_height = 16
        fb = driver.create_framebuffer(framebuffer_width, framebuffer_height)
        driver.bind_framebuffer(fb)

        # Clear the framebuffer to blue
        driver.clear_color(0.0, 0.0, 1.0, 1.0)
        driver.clear_depth(1.0)

        # Compile and link shaders
        vertex_shader_source = "attribute vec4 position; void main() { gl_Position = position; }"
        fragment_shader_source = "void main() { gl_FragColor = vec4(1.0, 0.0, 0.0, 1.0); }"

        vertex_shader = driver.compile_shader(vertex_shader_source, "vertex")
        fragment_shader = driver.compile_shader(fragment_shader_source, "fragment")
        
        shader_program = driver.graphics_api.link_program(vertex_shader, fragment_shader)
        driver.use_program(shader_program)

        # Create a vertex buffer (dummy data for a triangle)
        # Vertices are (x, y, z) in normalized device coordinates (-1 to 1)
        # For simplicity, we'll use screen coordinates for rasterizer, so these are just positions
        triangle_vertices = [
            0, 0, 0, # Vertex 1 (bottom-left)
            15, 0, 0, # Vertex 2 (bottom-right)
            7, 15, 0  # Vertex 3 (top-middle)
        ]
        # Convert to bytes for writing to memory
        # In a real scenario, this would be float data, but our memory manager uses bytes/ints
        # For simulation, we'll just pass the raw integers and let the rasterizer interpret
        vertex_buffer_data = triangle_vertices

        vertex_buffer_id = driver.create_buffer(len(vertex_buffer_data), "vertex")
        driver.buffer_data(vertex_buffer_id, vertex_buffer_data)

        # Draw the triangle
        # The rasterizer will interpret the dummy vertices and draw a simple shape
        driver.draw_arrays("triangles", 0, 3) # mode, first_vertex, count

        # Read back a portion of the color buffer to see results (simulated)
        # This will be the raw byte data from the simulated framebuffer
        color_buffer_content = driver.read_memory(fb["color"], framebuffer_width * framebuffer_height * 4)
        print("\n--- Simulated Framebuffer Content (first 64 bytes) ---")
        print(color_buffer_content[:64]) # Print first 64 bytes (16 pixels)
        print("-----------------------------------------------------")

    except Exception as e:
        print(f"An error occurred during graphics example: {e}")
    finally:
        driver.shutdown()

if __name__ == "__main__":
    run_simple_graphics_example()


